/*
    ClanRuby, A set of Ruby bindings for ClanLib

    Copyright (C) 2002 Russell Olsen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This is the ruby wrapper for the CL_Surface class.
 */

#include "clandefs.h"
#include "RB_ClanRuby.h"

/*
 * Ruby class for a surface.
 */
static VALUE cSurface;

/*
 * Free the underlying clanlib surface object. Used by the ruby garbase 
 * collector.
 */
static void freeSurface( CL_Surface *s)
{
	delete s;
}

/*
 * Create a ruby object to wrap the given surface.
 */
VALUE wrapSurface( CL_Surface *s )
{
	return Data_Wrap_Struct(cSurface, 0, freeSurface, s);
}

/*
 * Copy the surface to the screen at the location specified.
 */
static VALUE surface_putScreen(VALUE self, VALUE vX, VALUE vY)
{
	int x = NUM2INT(vX);
	int y = NUM2INT(vY);
	
	CL_Surface *s; 
	Data_Get_Struct( self, CL_Surface, s);
	
	s->put_screen( x, y );
	return Qnil;
}

/*
 * Get the dimensions of the surface.
 */
static VALUE surface_getWidth(VALUE self)
{
	CL_Surface *s; 
	Data_Get_Struct( self, CL_Surface, s);
	
	return INT2NUM(s->get_width());
}

static VALUE surface_getHeight(VALUE self)
{
	CL_Surface *s; 
	Data_Get_Struct( self, CL_Surface, s);
	
	return INT2NUM(s->get_height());
}

/*
 * Fill a rectangle.
 */
static VALUE fillRect(VALUE self, VALUE vX1, VALUE vY1, 
	VALUE vX2, VALUE vY2, VALUE vRed, VALUE vGreen, VALUE vBlue,
	VALUE vAlpha)
{
	
	CL_Surface *s; 
	Data_Get_Struct( self, CL_Surface, s);
	
	int x1 = NUM2INT(vX1);
	int y1 = NUM2INT(vY1);
	int x2 = NUM2INT(vX2);
	int y2 = NUM2INT(vY2);
	
	float r = (float)NUM2DBL(vRed);
	float g = (float)NUM2DBL(vGreen);
	float b = (float)NUM2DBL(vBlue);
	float alpha = (float)NUM2DBL(vAlpha);
	
	CL_SurfaceProvider *target = s->get_provider();

	target->fill_rect( x1, y1, x2, y2, r, g, b, alpha );
	return Qnil;
}

/*
 * Initialize the surface class for ruby.
 */
void initSurface()
{
	cSurface=rb_define_class_under(
		mClanRuby, 
		"Surface", 
		rb_cObject);


	rb_define_method(
		cSurface,
		"putScreen", 
		(VALFP)surface_putScreen,
		2);
		
	rb_define_method(
		cSurface,
		"width",
		(VALFP)surface_getWidth,
		0);
		
	rb_define_method(
		cSurface,
		"height", 
		(VALFP)surface_getHeight, 
		0);
		
	rb_define_method(
		cSurface, 
		"fillRect", 
		(VALFP)fillRect,
		8);
}
