/*
    ClanRuby, A set of Ruby bindings for ClanLib

    Copyright (C) 2002 Russell Olsen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This is the ruby wrapper for the CL_Rect class.
 */
#include "clandefs.h"
#include "RB_ClanRuby.h"


/*
 * The ruby class which wraps a CL_Rect object.
 */
static VALUE classRect;

/*
 * Free up the underlying CL_Rect object.
 */
static void freeRect( CL_Rect *s)
{
	delete s;
}


/*
 * Create a ruby rect object for the given clanlib object.
 */
VALUE wrapRect( CL_Rect *s )
{
	return Data_Wrap_Struct(classRect, 0, freeRect, s);
}

/*
 * Create a new ruby rect given the right, bottom, left and top coords.
 */
static VALUE rect_new(VALUE self, VALUE vR, VALUE vB, VALUE vL, VALUE vT)
{
	int r = NUM2INT(vR);
	int b = NUM2INT(vB);
	int l = NUM2INT(vL);
	int t = NUM2INT(vT);
	
	CL_Rect *s = new CL_Rect( r, b, l, t );
	VALUE ret = wrapRect(s);
	
	return ret;
}

/*
 * Move the rect by the given dx, dy.
 */
static VALUE rect_move(VALUE self, VALUE vDx, VALUE vDy)
{
	int dx = NUM2INT(vDx);
	int dy = NUM2INT(vDy);
	
	CL_Rect *s; 
	Data_Get_Struct( self, CL_Rect, s);
	
	s->move(dx, dy);
	
	return self;
}

/*
 * Return the dimension of this rect.
 */
static VALUE rect_getWidth(VALUE self)
{
	CL_Rect *s; 
	Data_Get_Struct( self, CL_Rect, s);
	
	return INT2NUM(s->get_width());
}

static VALUE rect_getHeight(VALUE self)
{
	CL_Rect *s; 
	Data_Get_Struct( self, CL_Rect, s);
	
	return INT2NUM(s->get_height());
}

/*
 * Return true if the point given is inside the rect.
 */
static VALUE rect_inside(VALUE self, VALUE vX, VALUE vY)
{
	int x = NUM2INT(vX);
	int y = NUM2INT(vY);
	
	CL_Rect *s; 
	Data_Get_Struct( self, CL_Rect, s);
	
	if ( s->inside(x, y) )
	{
		return Qtrue;
	}
	
	
	return Qfalse;
}

/*
 * Initialize the rect class for ruby.
 */
void initRect()
{
	classRect=rb_define_class_under(
		mClanRuby, 
		"Rect", 
		rb_cObject);
		
	rb_define_singleton_method(
		classRect, 
		"new", 
		(VALFP)rect_new,
		4);

	rb_define_method(
		classRect,
		"move", 
		(VALFP)rect_move,
		2);

		
	rb_define_method(
		classRect,
		"inside", 
		(VALFP)rect_inside,
		2);
		
	rb_define_method(
		classRect,
		"getWidth",
		(VALFP)rect_getWidth,
		0);

	rb_define_method(
		classRect,
		"getHeight",
		(VALFP)rect_getHeight,
		0);
		
	rb_define_method(
		classRect,
		"getHeight", 
		(VALFP)rect_getHeight, 
		0);
}
