/*
    ClanRuby, A set of Ruby bindings for ClanLib

    Copyright (C) 2002 Russell Olsen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This is the ruby wrapper for the CL_ClipRect class.
 */

#include "clandefs.h"
#include "RB_ClanRuby.h"


class RB_Handler
{
	public:
		/*
		 * Input slots for mouse stuff.
		 */
		CL_Slot _slot;
		
		/*
		 * Ruby method to call when a mouse is moved
		 */
		VALUE _rubyMethod;
		
		/*
		  * Constructor.
		  */
		RB_Handler()
		{
			_rubyMethod = Qnil;
		}
		

		virtual void connect( VALUE rubyMethod)
		{
			if ( rubyMethod != _rubyMethod )
			{
				disconnect();
			}
			
			if ( rubyMethod != Qnil )
			{
				connectSlot();
			
			}
			_rubyMethod = rubyMethod;
		}
		
		virtual void connectSlot() = 0;
		
		virtual void disconnect()
		{
			printf( "disconnecting\n" );
			
			if ( _rubyMethod != Qnil )
			{
				disconnectSlot();
			}
			
			_rubyMethod = Qnil;
		}
		
		virtual void disconnectSlot() = 0;
		
		virtual ~RB_Handler()
		{
			printf( "destroying handler, disconnecting\n" );
			disconnect();
		}
};

class RB_MouseMoveHandler: public RB_Handler
{
	public:
	
		RB_MouseMoveHandler(): 	RB_Handler() 
		{
		}
		
		/*
		 * Clanlib function which is called when a button is pressed,
		 * in turn calls the associated ruby button press method if there
		 * is one.
		 */
		void onMouseMove(int x, int y)
		{ 
			printf( "onMouseMove\n" );
			
			VALUE vX = INT2NUM(x);	
			VALUE vY = INT2NUM(y);
			ID callID = rb_intern("onMouseMove");
			rb_funcall( _rubyMethod, callID, 2, vX, vY );
			
			printf( "mouse move: %d %d\n", x, y );
		}
		
		virtual void connectSlot( )
		{
			_slot = CL_Mouse::sig_move.connect(
					this, &RB_MouseMoveHandler::onMouseMove);
		}
		
		void disconnectSlot()
		{
			CL_Mouse::sig_move.disconnect(_slot);
		}

};

class RB_MousePressHandler: public RB_Handler
{
	public:
	
		RB_MousePressHandler(): 	RB_Handler() 
		{
		}
		
		void onButtonPress(CL_Key key)
		{ 
			printf( "onButtonPress\n" );
			
		}
		
		virtual void connectSlot( )
		{
			_slot = CL_Mouse::sig_button_press.connect(
					this, &RB_MousePressHandler::onButtonPress);
		}
		
		void disconnectSlot()
		{
			CL_Mouse::sig_button_press.disconnect(_slot);
		}

};

/*
 * Ruby class.
 */
static VALUE cMouseHandler;

/*
 * Free the ClanLib object, used by the garbage collector.
 */
static void freeMouseHandler( RB_MouseMoveHandler *mh)
{
	printf( "deleting mouse handler\n");
	
	delete mh;
}

/*
 * Return a Ruby object which wraps the given clanlib object.
 */
VALUE wrapMouseHandler( RB_MouseMoveHandler *mh )
{
	return Data_Wrap_Struct(cMouseHandler, 0, freeMouseHandler, mh);
}

/*
 * Ruby constructor.
 */
static VALUE mousehandler_new(VALUE self)
{

	RB_MouseMoveHandler *mh = new RB_MouseMoveHandler( );
	VALUE ret = wrapMouseHandler(mh);
	
	return ret;
}

/*
 * Connect move event to ruby object.
 */
static VALUE mousehandler_listen(VALUE self, VALUE listener )
{
	RB_MouseMoveHandler *mh; 
	Data_Get_Struct( self, RB_MouseMoveHandler, mh);
	
	mh->connect(listener);
	
	return Qnil;
}


/*
 * Disconnect the handler.
 */
static VALUE mousehandler_unlisten(VALUE self )
{

	RB_MouseMoveHandler *mh; 
	Data_Get_Struct( self, RB_MouseMoveHandler, mh);
	
	mh->disconnect();

	
	return Qnil;
}

/*
 * Initialize the ruby class.
 */
void initMouseHandler()
{
	cMouseHandler=rb_define_class_under( mClanRuby, "MouseHandler", rb_cObject); 
	
	rb_define_singleton_method( cMouseHandler, "new", (VALFP)mousehandler_new, 0);
	rb_define_method( cMouseHandler, "connect", (VALFP)mousehandler_listen, 1);
	rb_define_method( cMouseHandler, "disconnect", (VALFP)mousehandler_unlisten, 0);

				
}
