/*
    ClanRuby, A set of Ruby bindings for ClanLib

    Copyright (C) 2002 Russell Olsen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This is the ruby wrapper for the CL_Input class.
 */
#include "clandefs.h"
#include "RB_ClanRuby.h"
#include <ClanLib/display.h>

/*
 * Input \slot used by ClanRuby.
 */
static CL_Slot buttonSlot;

/*
 * Ruby method to call when a button is pressed.
 */
static VALUE buttonPressHandler = Qnil;

/*
 * Clanlib function which is called when a button is pressed,
 * in turn calls the associated ruby button press method if there
 * is one.
 */
static void onButtonPress(CL_InputDevice *device, const CL_Key &key)
{
	VALUE vAscii = INT2NUM(key.ascii);
	VALUE vID = INT2NUM(key.id);	
	VALUE vX = rb_float_new(key.x);
	VALUE vY = rb_float_new(key.y);
	ID callID = rb_intern("call");
	
	if ( buttonPressHandler != Qnil )
	{
		rb_funcall( buttonPressHandler, callID, 
			4, vID, vAscii, vX, vY );
	}
}

/*
 * Ruby method to set the object which handles the button press.
 */
static void setButtonHandler(VALUE self, VALUE vHandler)
{
	buttonPressHandler = vHandler;
}

/*
 * Initialize the button press class.
 */
void initInput()
{
	/*
	 * Define the class...
	 */
	 VALUE cInput = rb_define_class_under( mClanRuby, "Input", rb_cObject);

	/*
	 * Connect the clanlib button handler to the ruby handler.
	 */
	buttonSlot = CL_Input::sig_button_press.connect( onButtonPress);
	
	rb_define_singleton_method( 
		cInput, "buttonHandler=", (VALFP)setButtonHandler, 1);		

	/*
	 * Define all the constants for the keys, and mouse buttons.
	 */
	RB_INT_CONST(mClanRuby,"CL_KEY_BACKSPACE",CL_KEY_BACKSPACE);
	RB_INT_CONST(mClanRuby,"CL_KEY_TAB",CL_KEY_TAB);
	RB_INT_CONST(mClanRuby,"CL_KEY_ENTER",CL_KEY_ENTER);
	RB_INT_CONST(mClanRuby,"CL_KEY_PAUSE",CL_KEY_PAUSE);
	RB_INT_CONST(mClanRuby,"CL_KEY_ESCAPE",CL_KEY_ESCAPE);
	RB_INT_CONST(mClanRuby,"CL_KEY_SPACE",CL_KEY_SPACE);
	
	RB_INT_CONST(mClanRuby, "CL_KEY_0", CL_KEY_0);
	RB_INT_CONST(mClanRuby, "CL_KEY_1", CL_KEY_1);
	RB_INT_CONST(mClanRuby, "CL_KEY_2", CL_KEY_2);
	RB_INT_CONST(mClanRuby, "CL_KEY_3", CL_KEY_3);
	RB_INT_CONST(mClanRuby, "CL_KEY_4", CL_KEY_4);
	RB_INT_CONST(mClanRuby, "CL_KEY_5", CL_KEY_5);
	RB_INT_CONST(mClanRuby, "CL_KEY_6", CL_KEY_6);
	RB_INT_CONST(mClanRuby, "CL_KEY_7", CL_KEY_7);
	RB_INT_CONST(mClanRuby, "CL_KEY_8", CL_KEY_8);
	RB_INT_CONST(mClanRuby, "CL_KEY_9", CL_KEY_9);
	
	RB_INT_CONST(mClanRuby, "CL_KEY_A", CL_KEY_A);
	RB_INT_CONST(mClanRuby, "CL_KEY_B", CL_KEY_B);
	RB_INT_CONST(mClanRuby, "CL_KEY_C", CL_KEY_C);
	RB_INT_CONST(mClanRuby, "CL_KEY_D", CL_KEY_D);
	RB_INT_CONST(mClanRuby, "CL_KEY_E", CL_KEY_E);
	RB_INT_CONST(mClanRuby, "CL_KEY_F", CL_KEY_F);
	RB_INT_CONST(mClanRuby, "CL_KEY_G", CL_KEY_G);
	RB_INT_CONST(mClanRuby, "CL_KEY_H", CL_KEY_H);
	RB_INT_CONST(mClanRuby, "CL_KEY_I", CL_KEY_I);
	RB_INT_CONST(mClanRuby, "CL_KEY_J", CL_KEY_J);
	RB_INT_CONST(mClanRuby, "CL_KEY_K", CL_KEY_K);
	RB_INT_CONST(mClanRuby, "CL_KEY_L", CL_KEY_L);
	RB_INT_CONST(mClanRuby, "CL_KEY_M", CL_KEY_M);
	RB_INT_CONST(mClanRuby, "CL_KEY_N", CL_KEY_N);
	RB_INT_CONST(mClanRuby, "CL_KEY_O", CL_KEY_O);
	RB_INT_CONST(mClanRuby, "CL_KEY_P", CL_KEY_P);
	RB_INT_CONST(mClanRuby, "CL_KEY_Q", CL_KEY_Q);
	RB_INT_CONST(mClanRuby, "CL_KEY_R", CL_KEY_R);
	RB_INT_CONST(mClanRuby, "CL_KEY_S", CL_KEY_S);
	RB_INT_CONST(mClanRuby, "CL_KEY_T", CL_KEY_T);
	RB_INT_CONST(mClanRuby, "CL_KEY_U", CL_KEY_U);
	RB_INT_CONST(mClanRuby, "CL_KEY_V", CL_KEY_V);
	RB_INT_CONST(mClanRuby, "CL_KEY_W", CL_KEY_W);
	RB_INT_CONST(mClanRuby, "CL_KEY_X", CL_KEY_X);
	RB_INT_CONST(mClanRuby, "CL_KEY_Y", CL_KEY_Y);
	RB_INT_CONST(mClanRuby, "CL_KEY_Z", CL_KEY_Z);
	
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_0", CL_KEY_0);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_1", CL_KEY_1);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_2", CL_KEY_2);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_3", CL_KEY_3);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_4", CL_KEY_4);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_5", CL_KEY_5);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_6", CL_KEY_6);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_7", CL_KEY_7);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_8", CL_KEY_8);
	RB_INT_CONST(mClanRuby, "CL_KEY_KP_9", CL_KEY_9);
	
	RB_INT_CONST(mClanRuby,"CL_KEY_KP_MULT",CL_KEY_KP_MULT);
	RB_INT_CONST(mClanRuby,"CL_KEY_KP_MINUS",CL_KEY_KP_MINUS);
	RB_INT_CONST(mClanRuby,"CL_KEY_KP_PLUS",CL_KEY_KP_PLUS);
	RB_INT_CONST(mClanRuby,"CL_KEY_KP_DECIMAL",CL_KEY_KP_DECIMAL);
	RB_INT_CONST(mClanRuby,"CL_KEY_KP_ENTER",CL_KEY_KP_ENTER);
	RB_INT_CONST(mClanRuby,"CL_KEY_UP",CL_KEY_UP);
	RB_INT_CONST(mClanRuby,"CL_KEY_DOWN",CL_KEY_DOWN);
	RB_INT_CONST(mClanRuby,"CL_KEY_RIGHT",CL_KEY_RIGHT);
	RB_INT_CONST(mClanRuby,"CL_KEY_LEFT",CL_KEY_LEFT);
	RB_INT_CONST(mClanRuby,"CL_KEY_DELETE",CL_KEY_DELETE);
	RB_INT_CONST(mClanRuby,"CL_KEY_INSERT",CL_KEY_INSERT);
	RB_INT_CONST(mClanRuby,"CL_KEY_HOME",CL_KEY_HOME);
	RB_INT_CONST(mClanRuby,"CL_KEY_END",CL_KEY_END);
	RB_INT_CONST(mClanRuby,"CL_KEY_PAGEUP",CL_KEY_PAGEUP);
	RB_INT_CONST(mClanRuby,"CL_KEY_PAGEDOWN",CL_KEY_PAGEDOWN);
	RB_INT_CONST(mClanRuby,"CL_KEY_F1",CL_KEY_F1);
	RB_INT_CONST(mClanRuby,"CL_KEY_F12",CL_KEY_F12);
	RB_INT_CONST(mClanRuby,"CL_KEY_NUMLOCK",CL_KEY_NUMLOCK);
	RB_INT_CONST(mClanRuby,"CL_KEY_CAPSLOCK",CL_KEY_CAPSLOCK);
	RB_INT_CONST(mClanRuby,"CL_KEY_SCRLOCK",CL_KEY_SCRLOCK);
	RB_INT_CONST(mClanRuby,"CL_KEY_RSHIFT",CL_KEY_RSHIFT);
	RB_INT_CONST(mClanRuby,"CL_KEY_LSHIFT",CL_KEY_LSHIFT);
	RB_INT_CONST(mClanRuby,"CL_KEY_RCTRL",CL_KEY_RCTRL);
	RB_INT_CONST(mClanRuby,"CL_KEY_LCTRL",CL_KEY_LCTRL);
	RB_INT_CONST(mClanRuby,"CL_KEY_ALTGR",CL_KEY_ALTGR);
	RB_INT_CONST(mClanRuby,"CL_KEY_ALT",CL_KEY_ALT);
	RB_INT_CONST(mClanRuby,"CL_KEY_PRINT",CL_KEY_PRINT);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_LEFTBUTTON",CL_MOUSE_LEFTBUTTON);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_MIDDLEBUTTON",CL_MOUSE_MIDDLEBUTTON);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_RIGHTBUTTON",CL_MOUSE_RIGHTBUTTON);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_THUMBBUTTON",CL_MOUSE_THUMBBUTTON);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_WHEELUP",CL_MOUSE_WHEELUP);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_WHEELDOWN",CL_MOUSE_WHEELDOWN);
	RB_INT_CONST(mClanRuby,"CL_MOUSE_WHEELBUTTON",CL_MOUSE_WHEELBUTTON);

}
