/*
    ClanRuby, A set of Ruby bindings for ClanLib

    Copyright (C) 2002 Russell Olsen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This is the ruby wrapper for the CL_Display class.
 */

#include "clandefs.h"
#include "RB_ClanRuby.h"

/*
 * Fill a rectangle.
 */
static VALUE fillRect(VALUE self, VALUE vX1, VALUE vY1, 
	VALUE vX2, VALUE vY2, VALUE vRed, VALUE vGreen, VALUE vBlue,
	VALUE vAlpha)
{
	int x1 = NUM2INT(vX1);
	int y1 = NUM2INT(vY1);
	int x2 = NUM2INT(vX2);
	int y2 = NUM2INT(vY2);
	
	float r = (float)NUM2DBL(vRed);
	float g = (float)NUM2DBL(vGreen);
	float b = (float)NUM2DBL(vBlue);
	float alpha = (float)NUM2DBL(vAlpha);
	
	CL_Display::fill_rect( x1, y1, x2, y2, r, g, b, alpha );
	return Qnil;
}

/*
 * Draw a line
 */
static VALUE drawLine(VALUE self, VALUE vX1, VALUE vY1, 
	VALUE vX2, VALUE vY2, VALUE vRed, VALUE vGreen, VALUE vBlue)
{
	int x1 = NUM2INT(vX1);
	int y1 = NUM2INT(vY1);
	int x2 = NUM2INT(vX2);
	int y2 = NUM2INT(vY2);
	
	float r = (float)NUM2DBL(vRed);
	float g = (float)NUM2DBL(vGreen);
	float b = (float)NUM2DBL(vBlue);
	
	CL_Display::draw_line( x1, y1, x2, y2, r, g, b );
	return Qnil;
}

/*
 * Flip the display, making the latest changes visible.
 */
static VALUE flipDisplay(VALUE self)
{
	CL_Display::flip_display();
	return Qnil;
}

/*
 * Set the video mode to the given size and depth.
 */
static VALUE setVideoMode( VALUE self, VALUE vW, VALUE vH, VALUE vD, VALUE vFS )
{
	int w = NUM2INT(vW);
	int h = NUM2INT(vH);
	int d = NUM2INT(vD);
	int fullScreen = (vFS == Qtrue);
	
	CL_Display::set_videomode(w, h, d, fullScreen);
	
	return Qnil;
}

/*
 * Push a clip rect onto the clipping stack.
 */
static VALUE pushClipRect( VALUE self, VALUE vRect )
{
	CL_ClipRect *r; 
	Data_Get_Struct( vRect, CL_ClipRect, r);
	
	CL_Display::push_clip_rect(*r);
	
	return vRect;
}

/*
 * Pop the current clip rect.
 */
static VALUE popClipRect( VALUE self, VALUE vRect )
{
	CL_Display::pop_clip_rect();
	
	return vRect;
}

/*
 * Push a translation onto the xform stack.
 */
static VALUE pushTranslateOffset(VALUE self, VALUE vX, VALUE vY ) 
{
	int x = NUM2INT(vX);
	int y = NUM2INT(vY);
	
	// printf( "offset %d %d\n", x, y );
	
	CL_Display::push_translate_offset( x, y);
	return Qnil;
}

/*
 * Pop the current xform.
 */
static VALUE popTranslateOffset( VALUE self )
{
	CL_Display::pop_translate_offset();
	
	return Qnil;
}

/*
 * Initialize the ruby class.
 */
void initDisplay()
{
	
	VALUE cDisplay=rb_define_class_under( mClanRuby, "Display", rb_cObject);

	rb_define_singleton_method( cDisplay, "setVideoMode", (VALFP)setVideoMode, 4);
		
	rb_define_singleton_method( cDisplay, "fillRect", (VALFP)fillRect, 8);
	
	rb_define_singleton_method( cDisplay, "drawLine", (VALFP)drawLine, 7);
	
	rb_define_singleton_method( cDisplay, "flipDisplay", (VALFP)flipDisplay, 0);
		
		
	rb_define_singleton_method( cDisplay, "pushTranslateOffset", 
		(VALFP)pushTranslateOffset, 2);
		
	rb_define_singleton_method( cDisplay, "popTranslateOffset", 
		(VALFP)popTranslateOffset, 0);
		
	rb_define_singleton_method( cDisplay, "pushClipRect", 
		(VALFP)pushClipRect, 1);
		
	rb_define_singleton_method( cDisplay, "popClipRect", 
		(VALFP)popClipRect, 0);
	
}
