/*
    ClanRuby, A set of Ruby bindings for ClanLib

    Copyright (C) 2002 Russell Olsen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This is the ruby wrapper for the CL_ClipRect class.
 */

#include "clandefs.h"
#include "RB_ClanRuby.h"


/*
 * Ruby class.
 */
static VALUE cClipRect;

/*
 * Free the ClanLib object, used by the garbage collector.
 */
static void freeClipRect( CL_ClipRect *r)
{
	delete r;
}

/*
 * Return a Ruby object which wraps the given clanlib object.
 */
VALUE wrapClipRect( CL_ClipRect *s )
{
	return Data_Wrap_Struct(cClipRect, 0, freeClipRect, s);
}

/*
 * Ruby constructor.
 */
static VALUE cliprect_new(VALUE self, VALUE vL, VALUE vT, VALUE vR, VALUE vB)
{
	int l = NUM2INT(vL);
	int t = NUM2INT(vT);
	int r = NUM2INT(vR);
	int b = NUM2INT(vB);

	
	CL_ClipRect *s = new CL_ClipRect( l, t, r, b );
	VALUE ret = wrapClipRect(s);
	
	return ret;
}

/*
 * Is the point inside of this rect?
 */
static VALUE cliprect_inside( VALUE self, VALUE vX, VALUE vY )
{

	int x = NUM2INT(vX);
	int y = NUM2INT(vY);
	
	CL_ClipRect *r; 
	Data_Get_Struct( self, CL_ClipRect, r);
	
	if ( (x >= r->m_x1) && (x < r->m_x2) )
	{
		if ( (y >= r->m_y1) && (y < r->m_y2) )
		{
			return Qtrue;
		}
	}
	return Qfalse;
}

/*
 * Accessor for left.
 */
static VALUE cliprect_getLeft(VALUE self)
{
	CL_ClipRect *s; 
	Data_Get_Struct( self, CL_ClipRect, s);
	
	return INT2NUM(s->m_x1);
}

/*
 * Accessor for top.
 */
static VALUE cliprect_getTop(VALUE self)
{
	CL_ClipRect *s; 
	Data_Get_Struct( self, CL_ClipRect, s);
	
	return INT2NUM(s->m_y1);
}

/*
 * Accessor for bottom.
 */
static VALUE cliprect_getBottom(VALUE self)
{
	CL_ClipRect *s; 
	Data_Get_Struct( self, CL_ClipRect, s);
	
	return INT2NUM(s->m_y2);
}

/*
 * Accessor for right.
 */
static VALUE cliprect_getRight(VALUE self)
{
	CL_ClipRect *s; 
	Data_Get_Struct( self, CL_ClipRect, s);
	
	return INT2NUM(s->m_x2);
}

/*
 * Accessor for width.
 */
static VALUE cliprect_getWidth(VALUE self)
{
	CL_ClipRect *s; 
	Data_Get_Struct( self, CL_ClipRect, s);
	
	return INT2NUM(s->m_x2 - s->m_x1);
}

/*
 * Accessor for ht.
 */
static VALUE cliprect_getHeight(VALUE self)
{
	CL_ClipRect *s; 
	Data_Get_Struct( self, CL_ClipRect, s);
	
	return INT2NUM(s->m_y2 - s->m_y1);
}

/*
 * Initialize the ruby class.
 */
void initClipRect()
{
	cClipRect=rb_define_class_under( mClanRuby, "ClipRect", rb_cObject); 
	
	rb_define_singleton_method( cClipRect, "new", (VALFP)cliprect_new, 4);

	rb_define_method(cClipRect, "inside", (VALFP)cliprect_inside, 2);
		
	rb_define_method(cClipRect, "left", (VALFP)cliprect_getLeft, 0);
		
	rb_define_method(cClipRect, "top", (VALFP)cliprect_getTop, 0);

	rb_define_method( cClipRect, "bottom", (VALFP)cliprect_getBottom, 0);
		
	rb_define_method( cClipRect, "right", (VALFP)cliprect_getRight, 0);
		
	rb_define_method( cClipRect, "width", (VALFP)cliprect_getWidth, 0);
		
	rb_define_method( cClipRect, "height", (VALFP)cliprect_getHeight, 0);
				
}
